/**
* \file: ScreenUtilsHooks.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: CarPLay
*
* \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*
* \copyright (c) 2013-2014 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

// This file is a modified copy of ScreenUtilsGStreamer.c
/*
	Copyright (C) 2013 Apple Inc. All Rights Reserved. Not to be used or disclosed without permission from Apple.
*/

#include "AirPlayHeaders.h"
#include <MiscUtils.h>

#include "Common.h"
#include "video/VideoChannel.h"
#include "utils/AutoCompletion.h"

#include "utils/debug.h"
#include "utils/Utils.h"

using namespace adit::carplay;

// public symbols
extern "C"
{
OSStatus ScreenStreamCreate(ScreenStreamRef* outStream);
OSStatus ScreenStreamInitialize(ScreenStreamRef inStream);
void ScreenStreamFinalize(ScreenStreamRef inStream);
OSStatus _ScreenStreamSetProperty(CFTypeRef inObject, CFObjectFlags inFlags, CFStringRef inProperty,
        CFTypeRef inQualifier, CFTypeRef inValue);
OSStatus ScreenStreamStart(ScreenStreamRef me);
void ScreenStreamStop(ScreenStreamRef me);
OSStatus ScreenStreamProcessData(ScreenStreamRef me, const uint8_t * inData, size_t inLen,
        uint64_t inDisplayTicks, CFDictionaryRef inOptions, ScreenStreamCompletion_f inCompletion,
        void * inContext);
}

//===========================================================================================================================
//	ScreenStream
//===========================================================================================================================

struct ScreenStreamPrivate
{
	CFRuntimeBase		base;					// CF type info. Must be first.
	uint8_t *			annexBHeaderPtr;		// Ptr to H.264 Annex-B header.
	size_t				annexBHeaderLen;		// Number of bytes in the H.264 Annex-B header.
	Boolean				annexBHeaderWritten;	// True if we've written the full Annex-B header to the decoder.
	Boolean				annexB;					// True if the source data is already in Annex-B format.
	size_t				nalSizeHeader;			// Number of bytes in the size before each NAL unit.
	int					widthPixels;			// Width of the screen in pixels.
	int					heightPixels;			// Height of the screen in pixels.
};

// ========== CFType type registration ==========

extern "C"
{
CFTypeID ScreenStreamGetTypeID(void);
static void _screenStreamGetTypeID(void* inContext);
static void _screenStreamFinalize(CFTypeRef inCF);
}

static dispatch_once_t _screenStreamInitOnce = 0;
static CFTypeID        _screenStreamTypeID   = _kCFRuntimeNotATypeID;
static const CFRuntimeClass kScreenStreamClass =
{
        0, // version
        "ScreenStream", // className
        nullptr, // init
        nullptr, // copy
        _screenStreamFinalize, // finalize
        nullptr, // equal -- NULL means pointer equality.
        nullptr, // hash  -- NULL means pointer hash.
        nullptr, // copyFormattingDesc
        nullptr, // copyDebugDesc
        nullptr, // reclaim
        nullptr // refcount
};

CFTypeID ScreenStreamGetTypeID(void)
{
	dispatch_once_f(&_screenStreamInitOnce, NULL, _screenStreamGetTypeID);
	return _screenStreamTypeID;
}

static void _screenStreamGetTypeID(void* inContext)
{
	(void)inContext;
	
	_screenStreamTypeID = _CFRuntimeRegisterClass(&kScreenStreamClass);
	if (_screenStreamTypeID == _kCFRuntimeNotATypeID)
	{
	    // TODO error handling
	}
}

static void _screenStreamFinalize(CFTypeRef inCF)
{
    auto me = (ScreenStreamPrivate*)(inCF);
    dipo_return_on_invalid_argument(dipo, me == nullptr);

    VideoChannel::Remove(me);
}

// ========== public functions ==========

OSStatus ScreenStreamCreate(ScreenStreamRef* outStream)
{
    if (outStream == nullptr)
        return kParamErr;

    ScreenStreamRef me;
    size_t extraLen = sizeof(*me) - sizeof(me->base);

    me = (ScreenStreamRef) _CFRuntimeCreateInstance(nullptr, ScreenStreamGetTypeID(),
            (CFIndex) extraLen, nullptr);
    dipo_exit_on_null(me);

    memset(((uint8_t*) me) + sizeof(me->base), 0, extraLen);

    auto vs = VideoChannel::Add(me);
    if (vs == nullptr)
    {
        // this should not happen; error is already logged
        VideoChannel::Remove(me);
        vs = VideoChannel::Add(me);
    }

    //TODO: For multi-session support get the appropriate session object.
    if (!vs->Initialize(Session::Get(nullptr), Server::Get(nullptr)->GetConfig()))
        return kUnknownErr;

    *outStream = me;
    return kNoErr;
}

OSStatus _ScreenStreamSetProperty(CFTypeRef inObject, CFObjectFlags inFlags, CFStringRef inProperty,
        CFTypeRef inQualifier, CFTypeRef inValue)
{
	ScreenStreamRef const		me = (ScreenStreamRef) inObject;
	OSStatus					err=kNoErr;
	
	(void) inFlags;
	(void) inQualifier;
	
	if( 0 ) {}
	
	// AVCC
	
	else if( CFEqual( inProperty, kScreenStreamProperty_AVCC ) )
	{
		const uint8_t *		avccPtr;
		size_t				avccLen;
		uint8_t *			headerPtr;
		size_t				headerLen;
		size_t				nalSizeHeader;
		
        if (!CFIsType( inValue, CFData ))
        {
            LOG_ERROR((dipo, "CF type check error in ScreenStreamSetProperty"));
        }
		
        avccPtr = CFDataGetBytePtr( (CFDataRef) inValue );
        avccLen = (size_t) CFDataGetLength( (CFDataRef) inValue );
        err = H264ConvertAVCCtoAnnexBHeader( avccPtr, avccLen, NULL, 0, &headerLen, NULL, NULL );
        if( err != kNoErr )
        {
            LOG_ERROR((dipo, "Error while H264 Conversion"));
        }
		
        headerPtr = (uint8_t *) malloc( headerLen );
        dipo_exit_on_null(headerPtr);
        err = H264ConvertAVCCtoAnnexBHeader( avccPtr, avccLen, headerPtr, headerLen, &headerLen, &nalSizeHeader, NULL );
            if ( !err )
            {
                if( me->annexBHeaderPtr ) free( me->annexBHeaderPtr );
                    me->annexBHeaderPtr		= headerPtr;
                    me->annexBHeaderLen		= headerLen;
                    me->nalSizeHeader		= nalSizeHeader;
                    me->annexBHeaderWritten = false;
            }
            else
            {
                free( headerPtr );
                LOG_ERROR((dipo, "Error while H264 Conversion while setting property to Screen stream"));
            }
    }
	// SourceFormat
    else if( CFEqual( inProperty, kScreenStreamProperty_SourceFormat ) )
    {
        if(      CFEqual( inValue, kScreenStreamFormat_H264AnnexB ) )	me->annexB = true;
        else if( CFEqual( inValue, kScreenStreamFormat_H264AVCC ) )		me->annexB = false;
        else { err = kUnsupportedErr;  }//goto exit;
    }
    // WidthPixels
    else if( CFEqual( inProperty, kScreenStreamProperty_WidthPixels ))
    {
        int screenWidth;
        CFNumberGetValue( (CFNumberRef) inValue, kCFNumberIntType, &screenWidth );
        me->widthPixels = screenWidth;
    }
    // HeightPixels
    else if( CFEqual( inProperty, kScreenStreamProperty_HeightPixels ))
    {
        int screenHeight;
        CFNumberGetValue( (CFNumberRef) inValue, kCFNumberIntType, &screenHeight );
        me->heightPixels = screenHeight;
    }
    //other
    else
    {
        err = kNotHandledErr;
        LOG_ERROR((dipo, "kNotHandledErr while setting propertyto Screen stream"));
    }

    return( err );
}

OSStatus ScreenStreamStart(ScreenStreamRef me)
{
    if (me == nullptr)
        return kParamErr;

    auto vs = VideoChannel::Get(me);
    if (vs == nullptr)
        return kUnknownErr;

    if (!vs->Start())
        return kUnknownErr;
    return kNoErr;
}

void ScreenStreamStop(ScreenStreamRef me)
{
    if (me == nullptr)
        return;

    auto vs = VideoChannel::Get(me);
    if (vs == nullptr)
        return;

    vs->Stop();
}

#if 1

// Annex B variant

OSStatus ScreenStreamProcessData(ScreenStreamRef me, const uint8_t* inData, size_t inLen,
        uint64_t inDisplayTicks, CFDictionaryRef inOptions, ScreenStreamCompletion_f inCompletion,
        void* inContext)
{
    if (me == nullptr || inData == nullptr)
        return kParamErr;

/*#if defined(DIPO_DEBUG_MEASURE_VIDEO) || defined(DIPO_DEBUG_OBSERVE_VIDEO)
    {
        uint64_t now = GetCurrentTimeNano();
#if DIPO_DEBUG_MEASURE_VIDEO
        printf("PERF %s %.14llu, timestamp: %llu (%.2f KB)\n", "ScreenStreamProcessData", now,
                inDisplayTicks, inLen / 1024.0f);
#endif
        debug_SetLastScreenProcessNano(now);
    }
#endif*/ // DIPO_DEBUG_MEASURE_VIDEO

    (void) inOptions;

    AutoCompletion autoCompletion(inCompletion, inContext);

    auto vs = VideoChannel::Get(me);
    if (vs == nullptr)
        return kNoErr; // return silently

    if (!me->annexB)
    {
        if (!me->annexBHeaderWritten)
        {
            vs->RenderScreen(me->annexBHeaderPtr, me->annexBHeaderLen, inDisplayTicks, 0,
                    (ScreenStreamCompletion_f)AutoCompletion::Completion,
                    autoCompletion.GetAdditionalReference());
            me->annexBHeaderWritten = true;
        }

        const uint8_t* src = inData;
        const uint8_t* end = src + inLen;
        const uint8_t* nalPtr;
        size_t nalLen;
        while (kNoErr == H264GetNextNALUnit(src, end, me->nalSizeHeader, &nalPtr, &nalLen, &src))
        {
            vs->RenderScreen((const uint8_t *) "\x00\x00\x00\x01", 4, inDisplayTicks, 0,
                    (ScreenStreamCompletion_f)AutoCompletion::Completion,
                    autoCompletion.GetAdditionalReference());

            const uint8_t* nalEnd = nalPtr + nalLen;
            const uint8_t* dataPtr;
            size_t dataLen;
            const uint8_t* suffixPtr;
            size_t suffixLen;
            while (H264EscapeEmulationPrevention(nalPtr, nalEnd, &dataPtr, &dataLen, &suffixPtr,
                    &suffixLen, &nalPtr))
            {
                if (dataLen > 0)
                {
                    vs->RenderScreen(dataPtr, dataLen, inDisplayTicks, 0,
                            (ScreenStreamCompletion_f)AutoCompletion::Completion,
                            autoCompletion.GetAdditionalReference());
                }
                if (suffixLen > 0)
                {
                    vs->RenderScreen(suffixPtr, suffixLen, inDisplayTicks, 0,
                            (ScreenStreamCompletion_f)AutoCompletion::Completion,
                            autoCompletion.GetAdditionalReference());
                }
            }

        }

            vs->Finish();

    }

    return kNoErr;
}
#elif 0

// Annex B / AVCC mix variant

OSStatus ScreenStreamProcessData(ScreenStreamRef me, const uint8_t* inData, size_t inLen,
        uint64_t inDisplayTicks, CFDictionaryRef inOptions, ScreenStreamCompletion_f inCompletion,
        void* inContext)
{
    if (me == nullptr || inData == nullptr)
        return kParamErr;

    (void) inOptions;

    AutoCompletion autoCompletion(inCompletion, inContext);

    auto vs = VideoChannel::Get(me);
    if (vs == nullptr)
        return kNoErr; // return silently

    if (!me->annexB)
    {
        if (!me->annexBHeaderWritten)
        {
            vs->RenderScreen(me->annexBHeaderPtr, me->annexBHeaderLen, inDisplayTicks, 0,
                    (ScreenStreamCompletion_f)AutoCompletion::Completion,
                    autoCompletion.GetAdditionalReference());
            me->annexBHeaderWritten = true;
        }

        vs->RenderScreen(inData, inLen, inDisplayTicks, 0,
                (ScreenStreamCompletion_f)AutoCompletion::Completion,
                autoCompletion.GetAdditionalReference());
    }

    return kNoErr;
}

#else

// AVCC variant

OSStatus ScreenStreamProcessData(ScreenStreamRef me, const uint8_t* inData, size_t inLen,
        uint64_t inDisplayTicks, CFDictionaryRef inOptions, ScreenStreamCompletion_f inCompletion,
        void* inContext)
{
    if (me == nullptr || inData == nullptr)
        return kParamErr;

    (void) inOptions;

    AutoCompletion autoCompletion(inCompletion, inContext);

    auto vs = VideoChannel::Get(me);
    if (vs == nullptr)
        return kNoErr; // return silently

    vs->RenderScreen(inData, inLen, inDisplayTicks, 0,
            (ScreenStreamCompletion_f)AutoCompletion::Completion,
            autoCompletion.GetAdditionalReference());

    return kNoErr;
}
#endif
